' Maze Maker for TwistyPassages
' Rev 1.0.0 William M Leue 16-Jan-2024

option default integer
option base 1

' Constants
const CELLH = 25
const CELLV = 20
const MINCR = 5
const MAXCOLS = (mm.hres-50)\CELLH
const MAXROWS = (mm.vres-50)\CELLV

const FWIDTH = 20
const FCOLOR = rgb(0, 128, 0)
const BGCOLOR = rgb(190, 184, 143)

const UP     = 128
const DOWN   = 129
const LEFT   = 130
const RIGHT  = 131
const SPACE  = 32
const INSERT = 132
const ENTER  = 13
const V      = 86
const ESC    = 27

' Globals
dim cells(MAXROWS, MAXCOLS)
'dim ncells(MAXROWS)
dim start(2) = (0, 0)
dim px = 0
dim py = 0
dim pwidth = 0
dim pheight = 0
dim mpath$ = ""
dim level = 0
dim message$ = ""
dim view = 0
dim existing = 0

' Main program
open "debug.txt" for output as #1
print "Edit Existing File? [Y,N]: ";
input "", a$
if LEFT$(UCASE$(a$), 1) = "Y" then
  exsiting = 1
  do
    ok = 1
    print "Enter level number: ";
    input "", fn$
    level = val(fn$)
    fn$ = "level" + str$(level)
    path$ = MakeFullPath$(fn$)
    ReadMaze path$, ok
  loop until ok
else
  do
    ok = 1
    print "Level Index? (1-?): ";
    input "", a$
    level = val(a$)
    if level < 1 then
      ok = 0
    end if
  loop until ok
  do
    path$ = MakeFullPath$("level" + str$(level))
    if TestExistingFile(path$) then
      print "The file '" + path$ + "' already exists! You cannot overwrite it!"
      end
    end if
    m$ = "Max width of maze [" + str$(MINCR) + "-" + str$(MAXCOLS) + "] "
    print m$;
    input "", a$
    pwidth = val(a$)
    if (pwidth < 5) or (pwidth > MAXCOLS) then ok = 0
  loop until ok
  do
    m$ = "Max height of maze [" + str$(MINCR) + "-" + str$(MAXROWS) + "] "
    print m$;
    input "", a$
    pheight = val(a$)
    if (pheight < MINCR) or (pheight > MAXROWS) then ok = 0
  loop until ok
  for row = 1 to pheight
    for col = 1 to pwidth
      cells(row, col) = 1
    next col
  next row
  SetLocation
end if
cls
DrawMaze
HandleEvents
end

' Test for an existing file to avoid accidental overwrite
function TestExistingFile(path$)
  on error skip 1
  open path$ for input as #4
  if mm.errno <> 0 then
    TestExistingFile = 0
    exit function
  else
    close #4
    TestExistingFile = 1
  end if
end function

' Add the prefix and suffix to the maze filename
function MakeFullPath$(fn$)
  if instr(1, fn$, "MAZES/") = 0 then
    fn$ = "./MAZES/" + fn$
  end if
  if instr(1, fn$, ".maz") = 0 then cat fn$, ".maz"
  MakeFullPath$ = fn$
end function

' Handle user keyboard inputs
sub HandleEvents
  local z$, cmd, row, col, ok
  row = 1 : col = 1
  HiliteCell row, col
  z$ = INKEY$
  do
    ok = 1
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(UCASE$(z$))
    select case cmd
      case UP
        if row > 1 then
          inc row, -1
        else
          row = pheight
        end if
      case DOWN
        if row < pheight then
          inc row
        else
          row = 1
        end if
      case LEFT
        if col > 1 then
          inc col, -1
        else
          col = pwidth
        end if
      case RIGHT
        if col < pwidth then
          inc col
        else
          col = 1
        end if
      case SPACE
        cells(row, col) = 1 - cells(row, col)
      case INSERT
        start(1) = row : start(2) = col
      case ENTER
        SaveMaze ok
        if ok then exit do
      case V
        view = 1 - view
        DrawMaze
      case ESC
        cls
        end
    end select
    DrawMaze
    HiliteCell row, col
  loop
end sub

' Save maze in file if conditons met
sub SaveMaze ok
  ok = 0
  if fn$ = "" then
    if start(1) = 0 then
      message$ =  "Cannot save to file - no START defined"
      exit sub
    end if
    message$ = ""
    cls
  path$ = MakeFullPath$("level" + str$(level))
  else
    text 0, mm.vres-1, space$(40), "LB"
  end if
  WriteMaze path$, ok
  cls
  print "Maze saved to '" + path$ + "'"
end sub

' Hilite the current cell with yellow outline
sub HiliteCell row, col
  local x, y
  x = px + (col-1)*CELLH
  y = py + (row-1)*CELLV
  box x, y, CELLH, CELLV,, rgb(yellow)
end sub

' Read a maze file
sub ReadMaze path$, ok
  local row, col, buf$
  ok = 1 
  on error skip 1
  open path$ for input as #2
  if mm.errno <> 0 then
    cls
    print "Open ";path$;" error: ";mm.errmsg$
    ok = 0
    exit sub
  end if
  line input #2, buf$
  level = val(buf$)
  line input #2, buf$
  cat buf$, ","
  pwidth = val(field$(buf$, 1, ","))
  pheight = val(field$(buf$, 2, ","))
  SetLocation  
  line input #2, buf$
  cat buf$, ","
  start(1) = val(field$(buf$, 1, ","))
  start(2) = val(field$(buf$, 2, ","))
  for row = 1 to pheight
    line input #2, buf$
    cat buf$, ","
    for col = 1 to pwidth
      cells(row, col) = val(field$(buf$, col, ","))
    next col
  next row
  close #2
end sub

' Write a Maze file
sub WriteMaze path$, ok
  local row, col
  ok = 1
  on error skip 1
  open path$ for output as #2
  if mm.errno <> 0 then
    cls
    print "Open ";path$;" error: ";mm.errmsg$
    ok = 1
    exit sub
  end if
  print #2, str$(level)
  print #2, str$(pwidth) + "," + str$(pheight)
  print #2, str$(start(1)) + "," + str$(start(2))
  for row = 1 to pheight
    for col = 1 to pwidth
      print #2, str$(cells(row, col)) + ",";
    next col
    print #2, ""
  next row
  close #2
end sub

' Draw the Maze
sub DrawMaze
  local row, col, x, y, ec, fc, v
  color rgb(white), rgb(black)
  page write 1
  cls
  DrawFrame 1
  for row = 1 to pheight
    y = py + (row-1)*CELLV
    for col = 1 to pwidth
      x = px + (col-1)*CELLH
      v = cells(row, col)
      if v then
        fc = rgb(black)
      else
        fc = BGCOLOR
      end if
      if view then
        ec = fc
      else
        c = rgb(gray)
      end if
      box x, y, CELLH, CELLV,, ec, fc
    next col
  next row
  if start(1) > 0 then
    x = px + (start(2)-1)*CELLH + CELLH\2
    y = py + (start(1)-1)*CELLV + CELLV\2
    circle x, y, CELLV\4,,, rgb(yellow), rgb(yellow)
  end if
  page write 0
  page copy 1 to 0, B
end sub

' Draw the Frame
sub DrawFrame inside
  local h, w, m$
  w = mm.hres : h = mm.vres
  box 0, 0, w, h, FWIDTH, FCOLOR
  if inside then
    box FWIDTH, FWIDTH, w-2*FWIDTH, h-2*FWIDTH,, BGCOLOR, BGCOLOR
  end if
  m$ = "LEVEL " + str$(level)
  text 30, 3, m$, "LT", 4,, rgb(white), -1
  text 31, 3, m$, "LT", 4,, rgb(white), -1
  m$ = "WIDTH " + str$(pwidth) + " HEIGHT " + str$(pheight)
  text w-FWIDTH-30, 3, m$, "RT", 4,, rgb(white), -1
  text w-FWIDTH-29, 3, m$, "RT", 4,, rgb(white), -1
  m$ = "MAKE MAZE"
  text 3, 60, m$, "LTV", 4,, rgb(white), -1
  m$ = "'V'=TOGGLE VIEW 'ENTER'=FINISH & WRITE MAZE 'INSERT'=SET START
  text FWIDTH+30, h-3, m$, "LB", 4,, rgb(white), -1
  DrawMessage
end sub

' set the maze tlc location
sub SetLocation
  px = mm.hres\2 - (pwidth*CELLH)\2
  py = mm.vres\2 - (pheight*CELLV)\2
end sub

' Draw Message
sub DrawMessage
  text mm.hres-5, mm.vres-3, message$, "RB", 4,, rgb(black), -1
  text mm.hres-5, mm.vres-3, message$, "RB", 4,, rgb(black), -1
end sub

